/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.jdbc.api.impl.arrow;

import com.databricks.jdbc.api.impl.arrow.AbstractArrowResultChunk;
import com.databricks.jdbc.api.impl.arrow.ArrowResultChunk;
import com.databricks.jdbc.api.impl.arrow.ChunkLinkFetcher;
import com.databricks.jdbc.api.impl.arrow.ChunkProvider;
import com.databricks.jdbc.api.impl.arrow.ChunkStatus;
import com.databricks.jdbc.api.impl.arrow.StreamingChunkDownloadTask;
import com.databricks.jdbc.common.CompressionCodec;
import com.databricks.jdbc.dbclient.IDatabricksHttpClient;
import com.databricks.jdbc.dbclient.impl.common.StatementId;
import com.databricks.jdbc.exception.DatabricksParsingException;
import com.databricks.jdbc.exception.DatabricksSQLException;
import com.databricks.jdbc.log.JdbcLogger;
import com.databricks.jdbc.log.JdbcLoggerFactory;
import com.databricks.jdbc.model.core.ChunkLinkFetchResult;
import com.databricks.jdbc.model.core.ExternalLink;
import com.databricks.jdbc.model.telemetry.enums.DatabricksDriverErrorCode;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nonnull;

public class StreamingChunkProvider
implements ChunkProvider {
    private static final JdbcLogger LOGGER = JdbcLoggerFactory.getLogger(StreamingChunkProvider.class);
    private static final String DOWNLOAD_THREAD_PREFIX = "databricks-jdbc-streaming-downloader-";
    private static final String PREFETCH_THREAD_NAME = "databricks-jdbc-link-prefetcher";
    private final int linkPrefetchWindow;
    private final int maxChunksInMemory;
    private final int chunkReadyTimeoutSeconds;
    private final ChunkLinkFetcher linkFetcher;
    private final IDatabricksHttpClient httpClient;
    private final CompressionCodec compressionCodec;
    private final StatementId statementId;
    private final double cloudFetchSpeedThreshold;
    private final ConcurrentMap<Long, ArrowResultChunk> chunks = new ConcurrentHashMap<Long, ArrowResultChunk>();
    private final AtomicLong currentChunkIndex = new AtomicLong(-1L);
    private final AtomicLong highestKnownChunkIndex = new AtomicLong(-1L);
    private volatile long nextLinkFetchIndex = 0L;
    private volatile long nextRowOffsetToFetch = 0L;
    private final AtomicLong nextDownloadIndex = new AtomicLong(0L);
    private volatile boolean endOfStreamReached = false;
    private volatile boolean closed = false;
    private volatile DatabricksSQLException prefetchError = null;
    private final AtomicLong totalRowCount = new AtomicLong(0L);
    private final ReentrantLock prefetchLock = new ReentrantLock();
    private final Condition consumerAdvanced = this.prefetchLock.newCondition();
    private final Condition chunkCreated = this.prefetchLock.newCondition();
    private final ReentrantLock downloadLock = new ReentrantLock();
    private final ExecutorService downloadExecutor;
    private final Thread linkPrefetchThread;
    private final AtomicInteger chunksInMemory = new AtomicInteger(0);

    public StreamingChunkProvider(ChunkLinkFetcher linkFetcher, IDatabricksHttpClient httpClient, CompressionCodec compressionCodec, StatementId statementId, int maxChunksInMemory, int linkPrefetchWindow, int chunkReadyTimeoutSeconds, double cloudFetchSpeedThreshold, ChunkLinkFetchResult initialLinks) throws DatabricksParsingException {
        this.linkFetcher = linkFetcher;
        this.httpClient = httpClient;
        this.compressionCodec = compressionCodec;
        this.statementId = statementId;
        this.maxChunksInMemory = maxChunksInMemory;
        this.linkPrefetchWindow = linkPrefetchWindow;
        this.chunkReadyTimeoutSeconds = chunkReadyTimeoutSeconds;
        this.cloudFetchSpeedThreshold = cloudFetchSpeedThreshold;
        LOGGER.info("Creating StreamingChunkProvider for statement {}: maxChunksInMemory={}, linkPrefetchWindow={}", statementId, maxChunksInMemory, linkPrefetchWindow);
        this.processInitialLinks(initialLinks);
        this.downloadExecutor = this.createDownloadExecutor(maxChunksInMemory);
        this.linkPrefetchThread = new Thread(this::linkPrefetchLoop, PREFETCH_THREAD_NAME);
        this.linkPrefetchThread.setDaemon(true);
        this.linkPrefetchThread.start();
        this.triggerDownloads();
        this.notifyConsumerAdvanced();
    }

    @Override
    public boolean hasNextChunk() {
        if (this.closed) {
            return false;
        }
        if (!this.endOfStreamReached) {
            return true;
        }
        return this.currentChunkIndex.get() < this.highestKnownChunkIndex.get();
    }

    @Override
    public boolean next() throws DatabricksSQLException {
        if (this.closed) {
            return false;
        }
        long prevIndex = this.currentChunkIndex.get();
        if (prevIndex >= 0L) {
            this.releaseChunk(prevIndex);
        }
        if (!this.hasNextChunk()) {
            return false;
        }
        this.currentChunkIndex.incrementAndGet();
        this.notifyConsumerAdvanced();
        return true;
    }

    @Override
    public AbstractArrowResultChunk getChunk() throws DatabricksSQLException {
        long chunkIdx = this.currentChunkIndex.get();
        if (chunkIdx < 0L) {
            return null;
        }
        ArrowResultChunk chunk = (ArrowResultChunk)this.chunks.get(chunkIdx);
        if (chunk == null) {
            LOGGER.debug("Chunk {} not yet available, waiting for prefetch", chunkIdx);
            this.waitForChunkCreation(chunkIdx);
            chunk = (ArrowResultChunk)this.chunks.get(chunkIdx);
        }
        if (chunk == null) {
            throw new DatabricksSQLException("Chunk " + chunkIdx + " not found after waiting", DatabricksDriverErrorCode.CHUNK_READY_ERROR);
        }
        try {
            chunk.waitForChunkReady();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new DatabricksSQLException("Interrupted waiting for chunk " + chunkIdx, (Throwable)e, DatabricksDriverErrorCode.THREAD_INTERRUPTED_ERROR);
        }
        catch (ExecutionException e) {
            throw new DatabricksSQLException("Failed to prepare chunk " + chunkIdx, e.getCause(), DatabricksDriverErrorCode.CHUNK_READY_ERROR);
        }
        catch (TimeoutException e) {
            throw new DatabricksSQLException("Timeout waiting for chunk " + chunkIdx + " (timeout: " + this.chunkReadyTimeoutSeconds + "s)", DatabricksDriverErrorCode.CHUNK_READY_ERROR);
        }
        return chunk;
    }

    @Override
    public void close() {
        if (this.closed) {
            return;
        }
        LOGGER.info("Closing StreamingChunkProvider for statement {}", this.statementId);
        this.closed = true;
        this.notifyConsumerAdvanced();
        this.notifyChunkCreated();
        if (this.linkPrefetchThread != null) {
            this.linkPrefetchThread.interrupt();
        }
        if (this.downloadExecutor != null) {
            this.downloadExecutor.shutdownNow();
        }
        for (ArrowResultChunk chunk : this.chunks.values()) {
            try {
                chunk.releaseChunk();
            }
            catch (Exception e) {
                LOGGER.warn("Error releasing chunk: {}", e.getMessage());
            }
        }
        this.chunks.clear();
        if (this.linkFetcher != null) {
            this.linkFetcher.close();
        }
    }

    @Override
    public long getRowCount() {
        return this.totalRowCount.get();
    }

    @Override
    public long getChunkCount() {
        if (this.endOfStreamReached) {
            return this.highestKnownChunkIndex.get() + 1L;
        }
        return -1L;
    }

    @Override
    public boolean isClosed() {
        return this.closed;
    }

    private void linkPrefetchLoop() {
        LOGGER.debug("Link prefetch thread started for statement {}", this.statementId);
        while (!this.closed && !Thread.currentThread().isInterrupted()) {
            try {
                this.prefetchLock.lock();
                try {
                    long targetIndex = this.currentChunkIndex.get() + (long)this.linkPrefetchWindow;
                    while (!this.endOfStreamReached && this.nextLinkFetchIndex > targetIndex) {
                        if (this.closed) {
                            break;
                        }
                        LOGGER.debug("Prefetch caught up, waiting for consumer. next={}, target={}", this.nextLinkFetchIndex, targetIndex);
                        this.consumerAdvanced.await();
                        targetIndex = this.currentChunkIndex.get() + (long)this.linkPrefetchWindow;
                    }
                }
                finally {
                    this.prefetchLock.unlock();
                }
                if (this.closed || this.endOfStreamReached) break;
                this.fetchNextLinkBatch();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.debug("Link prefetch thread interrupted");
                break;
            }
            catch (DatabricksSQLException e) {
                LOGGER.error("Error fetching links: {}", e.getMessage());
                this.prefetchError = e;
                this.notifyChunkCreated();
                break;
            }
            catch (Exception e) {
                LOGGER.error("Unexpected error in link prefetch: {}", e.getMessage(), e);
                this.prefetchError = new DatabricksSQLException("Unexpected error in link prefetch: " + e.getMessage(), (Throwable)e, DatabricksDriverErrorCode.CHUNK_READY_ERROR);
                this.notifyChunkCreated();
                break;
            }
        }
        LOGGER.debug("Link prefetch thread exiting for statement {}", this.statementId);
    }

    private void fetchNextLinkBatch() throws DatabricksSQLException {
        if (this.endOfStreamReached || this.closed) {
            return;
        }
        LOGGER.debug("Fetching links starting from index {}, row offset {} for statement {}", this.nextLinkFetchIndex, this.nextRowOffsetToFetch, this.statementId);
        ChunkLinkFetchResult result = this.linkFetcher.fetchLinks(this.nextLinkFetchIndex, this.nextRowOffsetToFetch);
        if (result.isEndOfStream()) {
            LOGGER.info("End of stream reached for statement {}", this.statementId);
            this.endOfStreamReached = true;
            return;
        }
        for (ExternalLink link : result.getChunkLinks()) {
            this.createChunkFromLink(link);
        }
        if (result.hasMore()) {
            this.nextLinkFetchIndex = result.getNextFetchIndex();
            this.nextRowOffsetToFetch = result.getNextRowOffset();
        } else {
            this.endOfStreamReached = true;
            LOGGER.info("End of stream reached for statement {} (hasMore=false)", this.statementId);
        }
        this.triggerDownloads();
    }

    private void processInitialLinks(ChunkLinkFetchResult initialLinks) throws DatabricksParsingException {
        if (initialLinks == null) {
            LOGGER.debug("No initial links provided for statement {}", this.statementId);
            return;
        }
        LOGGER.info("Processing {} initial links for statement {}", initialLinks.getChunkLinks().size(), this.statementId);
        for (ExternalLink link : initialLinks.getChunkLinks()) {
            this.createChunkFromLink(link);
        }
        if (initialLinks.hasMore()) {
            this.nextLinkFetchIndex = initialLinks.getNextFetchIndex();
            this.nextRowOffsetToFetch = initialLinks.getNextRowOffset();
            LOGGER.debug("Next fetch position set to chunk index {}, row offset {} from initial links", this.nextLinkFetchIndex, this.nextRowOffsetToFetch);
        } else {
            this.endOfStreamReached = true;
            LOGGER.info("End of stream reached from initial links for statement {}", this.statementId);
        }
    }

    private void createChunkFromLink(ExternalLink link) throws DatabricksParsingException {
        long chunkIndex = link.getChunkIndex();
        if (this.chunks.containsKey(chunkIndex)) {
            LOGGER.debug("Chunk {} already exists, skipping creation", chunkIndex);
            return;
        }
        long rowCount = link.getRowCount();
        long rowOffset = link.getRowOffset();
        ArrowResultChunk chunk = ArrowResultChunk.builder().withStatementId(this.statementId).withChunkMetadata(chunkIndex, rowCount, rowOffset).withChunkReadyTimeoutSeconds(this.chunkReadyTimeoutSeconds).build();
        chunk.setChunkLink(link);
        this.chunks.put(chunkIndex, chunk);
        this.highestKnownChunkIndex.updateAndGet(current -> Math.max(current, chunkIndex));
        this.totalRowCount.addAndGet(rowCount);
        this.notifyChunkCreated();
        LOGGER.debug("Created chunk {} with {} rows for statement {}", chunkIndex, rowCount, this.statementId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void triggerDownloads() {
        this.downloadLock.lock();
        try {
            long downloadIdx = this.nextDownloadIndex.get();
            while (!this.closed && this.chunksInMemory.get() < this.maxChunksInMemory && downloadIdx <= this.highestKnownChunkIndex.get()) {
                ArrowResultChunk chunk = (ArrowResultChunk)this.chunks.get(downloadIdx);
                if (chunk == null) {
                    break;
                }
                ChunkStatus status = chunk.getStatus();
                if (status == ChunkStatus.PENDING || status == ChunkStatus.URL_FETCHED) {
                    this.submitDownloadTask(chunk);
                    this.chunksInMemory.incrementAndGet();
                }
                downloadIdx = this.nextDownloadIndex.incrementAndGet();
            }
        }
        finally {
            this.downloadLock.unlock();
        }
    }

    private void submitDownloadTask(ArrowResultChunk chunk) {
        LOGGER.debug("Submitting download task for chunk {}", chunk.getChunkIndex());
        StreamingChunkDownloadTask task = new StreamingChunkDownloadTask(chunk, this.httpClient, this.compressionCodec, this.linkFetcher, this.cloudFetchSpeedThreshold);
        this.downloadExecutor.submit(task);
    }

    private void releaseChunk(long chunkIndex) {
        ArrowResultChunk chunk = (ArrowResultChunk)this.chunks.get(chunkIndex);
        if (chunk != null && chunk.releaseChunk()) {
            this.chunks.remove(chunkIndex);
            this.chunksInMemory.decrementAndGet();
            LOGGER.debug("Released chunk {}, chunksInMemory={}", chunkIndex, this.chunksInMemory.get());
            this.triggerDownloads();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void waitForChunkCreation(long chunkIndex) throws DatabricksSQLException {
        this.prefetchLock.lock();
        try {
            while (!this.closed && !this.chunks.containsKey(chunkIndex)) {
                if (this.prefetchError != null) {
                    throw new DatabricksSQLException("Link prefetch failed: " + this.prefetchError.getMessage(), (Throwable)this.prefetchError, DatabricksDriverErrorCode.CHUNK_READY_ERROR);
                }
                long highestKnown = this.highestKnownChunkIndex.get();
                if (this.endOfStreamReached && chunkIndex > highestKnown) {
                    throw new DatabricksSQLException("Chunk " + chunkIndex + " does not exist (highest known: " + highestKnown + ")", DatabricksDriverErrorCode.CHUNK_READY_ERROR);
                }
                try {
                    this.chunkCreated.await();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new DatabricksSQLException("Interrupted waiting for chunk creation", (Throwable)e, DatabricksDriverErrorCode.THREAD_INTERRUPTED_ERROR);
                    return;
                }
            }
        }
        finally {
            this.prefetchLock.unlock();
        }
    }

    private void notifyConsumerAdvanced() {
        this.prefetchLock.lock();
        try {
            this.consumerAdvanced.signalAll();
        }
        finally {
            this.prefetchLock.unlock();
        }
    }

    private void notifyChunkCreated() {
        this.prefetchLock.lock();
        try {
            this.chunkCreated.signalAll();
        }
        finally {
            this.prefetchLock.unlock();
        }
    }

    private ExecutorService createDownloadExecutor(int poolSize) {
        ThreadFactory threadFactory = new ThreadFactory(){
            private final AtomicInteger threadCount = new AtomicInteger(1);

            @Override
            public Thread newThread(@Nonnull Runnable r) {
                Thread thread = new Thread(r);
                thread.setName(StreamingChunkProvider.DOWNLOAD_THREAD_PREFIX + this.threadCount.getAndIncrement());
                thread.setDaemon(true);
                return thread;
            }
        };
        return Executors.newFixedThreadPool(poolSize, threadFactory);
    }
}

