/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.jdbc.dbclient.impl.common;

import com.databricks.internal.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import com.databricks.internal.sdk.WorkspaceClient;
import com.databricks.internal.sdk.core.CredentialsProvider;
import com.databricks.internal.sdk.core.DatabricksConfig;
import com.databricks.internal.sdk.core.DatabricksException;
import com.databricks.internal.sdk.core.GoogleCredentialsCredentialsProvider;
import com.databricks.internal.sdk.core.GoogleIdCredentialsProvider;
import com.databricks.internal.sdk.core.PatCredentialsProvider;
import com.databricks.internal.sdk.core.ProxyConfig;
import com.databricks.internal.sdk.core.commons.CommonsHttpClient;
import com.databricks.internal.sdk.core.oauth.AzureServicePrincipalCredentialsProvider;
import com.databricks.internal.sdk.core.oauth.ExternalBrowserCredentialsProvider;
import com.databricks.internal.sdk.core.oauth.OAuthM2MServicePrincipalCredentialsProvider;
import com.databricks.internal.sdk.core.oauth.TokenCache;
import com.databricks.internal.sdk.core.utils.Cloud;
import com.databricks.jdbc.api.internal.IDatabricksConnectionContext;
import com.databricks.jdbc.auth.AzureMSICredentialProvider;
import com.databricks.jdbc.auth.DatabricksTokenFederationProvider;
import com.databricks.jdbc.auth.EncryptedFileTokenCache;
import com.databricks.jdbc.auth.NoOpTokenCache;
import com.databricks.jdbc.auth.OAuthRefreshCredentialsProvider;
import com.databricks.jdbc.auth.PrivateKeyClientCredentialProvider;
import com.databricks.jdbc.common.AuthMech;
import com.databricks.jdbc.common.util.DatabricksAuthUtil;
import com.databricks.jdbc.common.util.DriverUtil;
import com.databricks.jdbc.dbclient.impl.common.ConfiguratorUtils;
import com.databricks.jdbc.exception.DatabricksParsingException;
import com.databricks.jdbc.exception.DatabricksSSLException;
import com.databricks.jdbc.exception.DatabricksValidationException;
import com.databricks.jdbc.log.JdbcLogger;
import com.databricks.jdbc.log.JdbcLoggerFactory;
import com.databricks.jdbc.model.telemetry.enums.DatabricksDriverErrorCode;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

public class ClientConfigurator {
    private static final JdbcLogger LOGGER = JdbcLoggerFactory.getLogger(ClientConfigurator.class);
    private final IDatabricksConnectionContext connectionContext;
    private DatabricksConfig databricksConfig;

    public ClientConfigurator(IDatabricksConnectionContext connectionContext) throws DatabricksSSLException, DatabricksValidationException {
        this.connectionContext = connectionContext;
        this.databricksConfig = new DatabricksConfig();
        this.databricksConfig.setDisableOauthRefreshToken(connectionContext.getDisableOauthRefreshToken());
        CommonsHttpClient.Builder httpClientBuilder = new CommonsHttpClient.Builder();
        httpClientBuilder.withTimeoutSeconds(connectionContext.getSocketTimeout());
        this.setupProxyConfig(httpClientBuilder);
        this.setupConnectionManager(httpClientBuilder);
        this.databricksConfig.setHttpClient(httpClientBuilder.build());
        this.setupDiscoveryEndpoint();
        this.setupAuthConfig();
        this.databricksConfig.resolve();
    }

    public static Path getTokenCachePath(String host, String clientId, List<String> scopes) {
        String userHome = System.getProperty("user.home");
        Path homeDir = Paths.get(userHome, new String[0]);
        Path databricksDir = homeDir.resolve(".config/databricks-jdbc/oauth");
        String uniqueIdentifier = ClientConfigurator.createUniqueIdentifier(host, clientId, scopes);
        String filename = "token-cache-" + uniqueIdentifier;
        return databricksDir.resolve(filename);
    }

    private static String createUniqueIdentifier(String host, String clientId, List<String> scopes) {
        host = host != null ? host : "";
        clientId = clientId != null ? clientId : "";
        scopes = scopes != null ? scopes : List.of();
        String combined = host + ";" + clientId + ";" + String.join((CharSequence)",", scopes);
        int hash = combined.hashCode();
        return Integer.toHexString(hash & Integer.MAX_VALUE);
    }

    void setupConnectionManager(CommonsHttpClient.Builder httpClientBuilder) throws DatabricksSSLException, DatabricksValidationException {
        PoolingHttpClientConnectionManager connManager = ConfiguratorUtils.getBaseConnectionManager(this.connectionContext);
        connManager.setMaxTotal(this.connectionContext.getHttpConnectionPoolSize());
        connManager.setDefaultMaxPerRoute(this.connectionContext.getHttpMaxConnectionsPerRoute());
        httpClientBuilder.withConnectionManager(connManager);
    }

    public void setupProxyConfig(CommonsHttpClient.Builder httpClientBuilder) {
        ProxyConfig proxyConfig = new ProxyConfig().setUseSystemProperties(this.connectionContext.getUseSystemProxy());
        if (this.connectionContext.getUseProxy().booleanValue()) {
            proxyConfig.setHost(this.connectionContext.getProxyHost()).setPort(this.connectionContext.getProxyPort());
        }
        if (this.connectionContext.getUseProxy().booleanValue() || this.connectionContext.getUseSystemProxy().booleanValue()) {
            proxyConfig.setUsername(this.connectionContext.getProxyUser()).setPassword(this.connectionContext.getProxyPassword()).setProxyAuthType(this.connectionContext.getProxyAuthType()).setNonProxyHosts(ClientConfigurator.convertNonProxyHostConfigToBeSystemPropertyCompliant(this.connectionContext.getNonProxyHosts()));
        }
        httpClientBuilder.withProxyConfig(proxyConfig);
    }

    public WorkspaceClient getWorkspaceClient() {
        return new WorkspaceClient(this.databricksConfig);
    }

    public void setupAuthConfig() {
        AuthMech authMech = this.connectionContext.getAuthMech();
        try {
            switch (authMech) {
                case OAUTH: {
                    this.setupOAuthConfig();
                    break;
                }
                default: {
                    this.setupAccessTokenConfig();
                    break;
                }
            }
        }
        catch (DatabricksParsingException e) {
            String errorMessage = "Error while parsing auth config";
            LOGGER.error(errorMessage);
            throw new DatabricksException(errorMessage, e);
        }
    }

    public void setupOAuthConfig() throws DatabricksParsingException {
        switch (this.connectionContext.getAuthFlow()) {
            case TOKEN_PASSTHROUGH: {
                if (this.connectionContext.getOAuthRefreshToken() != null) {
                    this.setupU2MRefreshConfig();
                    break;
                }
                this.setupOAuthAccessTokenConfig();
                break;
            }
            case CLIENT_CREDENTIALS: {
                this.setupM2MConfig();
                break;
            }
            case BROWSER_BASED_AUTHENTICATION: {
                this.setupU2MConfig();
                break;
            }
            case AZURE_MANAGED_IDENTITIES: {
                this.setupAzureMI();
            }
        }
    }

    public void setupU2MConfig() throws DatabricksParsingException {
        TokenCache tokenCache;
        int redirectPort = this.findAvailablePort(this.connectionContext.getOAuth2RedirectUrlPorts());
        String redirectUrl = String.format("http://localhost:%d", redirectPort);
        String host = this.connectionContext.getHostForOAuth();
        String clientId = this.connectionContext.getClientId();
        this.databricksConfig.setAuthType("external-browser").setHost(host).setClientId(clientId).setOAuthBrowserAuthTimeout(Duration.ofHours(1L)).setClientSecret(this.connectionContext.getClientSecret()).setOAuthRedirectUrl(redirectUrl);
        LOGGER.info("Using OAuth redirect URL: {}", redirectUrl);
        this.databricksConfig.setScopes(this.connectionContext.getOAuthScopesForU2M());
        if (this.connectionContext.isTokenCacheEnabled()) {
            if (this.connectionContext.getTokenCachePassPhrase() == null) {
                LOGGER.error("No token cache passphrase configured");
                throw new DatabricksException("No token cache passphrase configured");
            }
            Path tokenCachePath = ClientConfigurator.getTokenCachePath(host, clientId, this.databricksConfig.getScopes());
            tokenCache = new EncryptedFileTokenCache(tokenCachePath, this.connectionContext.getTokenCachePassPhrase());
        } else {
            tokenCache = new NoOpTokenCache();
        }
        this.databricksConfig.setCredentialsProvider(this.wrapWithTokenFederationIfEnabled(new ExternalBrowserCredentialsProvider(tokenCache)));
    }

    public int findAvailablePort(List<Integer> initialPorts) {
        List<Integer> portsToTry;
        if (initialPorts.size() == 1) {
            int startPort = initialPorts.get(0);
            int maxAttempts = 20;
            portsToTry = new ArrayList<Integer>(maxAttempts);
            for (int i = 0; i < maxAttempts; ++i) {
                portsToTry.add(startPort + i);
            }
            LOGGER.debug("Single port provided {}, will try ports {} through {}", startPort, startPort, startPort + maxAttempts - 1);
        } else {
            portsToTry = initialPorts;
            LOGGER.debug("Multiple ports provided, will try: {}", portsToTry);
        }
        for (int port : portsToTry) {
            if (this.isPortAvailable(port)) {
                return port;
            }
            LOGGER.debug("Port {} is not available, trying next port", port);
        }
        LOGGER.error("No available ports found among: {}", portsToTry);
        throw new DatabricksException("No available port found for OAuth redirect URL. Tried ports: " + String.valueOf(portsToTry));
    }

    boolean isPortAvailable(int port) {
        boolean bl;
        ServerSocket serverSocket = new ServerSocket();
        try {
            serverSocket.setReuseAddress(true);
            serverSocket.bind(new InetSocketAddress(port));
            bl = true;
        }
        catch (Throwable throwable) {
            try {
                try {
                    serverSocket.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                return false;
            }
        }
        serverSocket.close();
        return bl;
    }

    public void setupAccessTokenConfig() throws DatabricksParsingException {
        this.databricksConfig.setAuthType("pat").setHost(this.connectionContext.getHostUrl()).setToken(this.connectionContext.getToken());
    }

    public void setupOAuthAccessTokenConfig() throws DatabricksParsingException {
        if (DatabricksAuthUtil.isTokenJWT(this.connectionContext.getPassThroughAccessToken()).booleanValue()) {
            CredentialsProvider credentialsProvider = this.wrapWithTokenFederationIfEnabled(new PatCredentialsProvider());
            this.databricksConfig.setCredentialsProvider(credentialsProvider);
        }
        this.databricksConfig.setAuthType("pat").setHost(this.connectionContext.getHostUrl()).setToken(this.connectionContext.getPassThroughAccessToken());
    }

    public void resetAccessTokenInConfig(String newAccessToken) {
        this.databricksConfig = DatabricksAuthUtil.initializeConfigWithToken(newAccessToken, this.databricksConfig);
        this.databricksConfig.resolve();
    }

    public void setupU2MRefreshConfig() throws DatabricksParsingException {
        this.databricksConfig.setHost(this.connectionContext.getHostForOAuth()).setClientId(this.connectionContext.getClientId()).setClientSecret(this.connectionContext.getClientSecret());
        OAuthRefreshCredentialsProvider provider = new OAuthRefreshCredentialsProvider(this.connectionContext, this.databricksConfig);
        CredentialsProvider wrappedProvider = this.wrapWithTokenFederationIfEnabled(provider);
        this.databricksConfig.setAuthType(wrappedProvider.authType()).setCredentialsProvider(wrappedProvider);
    }

    public void setupM2MConfig() throws DatabricksParsingException {
        if (DriverUtil.isRunningAgainstFake()) {
            this.databricksConfig.setHost(this.connectionContext.getHostUrl());
        } else {
            this.databricksConfig.setHost(this.connectionContext.getHostForOAuth());
        }
        if (this.connectionContext.getCloud() == Cloud.GCP && !this.connectionContext.getGcpAuthType().equals("oauth-m2m")) {
            String authType = this.connectionContext.getGcpAuthType();
            this.databricksConfig.setAuthType(authType);
            if (authType.equals("google-credentials")) {
                this.databricksConfig.setGoogleCredentials(this.connectionContext.getGoogleCredentials());
                this.databricksConfig.setCredentialsProvider(this.wrapWithTokenFederationIfEnabled(new GoogleCredentialsCredentialsProvider()));
            } else {
                this.databricksConfig.setGoogleServiceAccount(this.connectionContext.getGoogleServiceAccount());
                this.databricksConfig.setCredentialsProvider(this.wrapWithTokenFederationIfEnabled(new GoogleIdCredentialsProvider()));
            }
        } else if (this.connectionContext.getAzureTenantId() != null) {
            LOGGER.debug("Using Azure Active Directory (AAD) Service Principal OAuth");
            if (this.connectionContext.getCloud() != Cloud.AZURE) {
                throw new DatabricksParsingException("Azure client credentials flow is only supported for Azure cloud", DatabricksDriverErrorCode.UNSUPPORTED_OPERATION);
            }
            this.databricksConfig.setAuthType("azure-client-secret").setAzureClientId(this.connectionContext.getClientId()).setAzureClientSecret(this.connectionContext.getClientSecret()).setAzureTenantId(this.connectionContext.getAzureTenantId()).setCredentialsProvider(this.wrapWithTokenFederationIfEnabled(new AzureServicePrincipalCredentialsProvider()));
        } else {
            this.databricksConfig.setClientId(this.connectionContext.getClientId()).setClientSecret(this.connectionContext.getClientSecret());
            if (this.connectionContext.useJWTAssertion()) {
                PrivateKeyClientCredentialProvider jwtProvider = new PrivateKeyClientCredentialProvider(this.connectionContext, this.databricksConfig);
                this.databricksConfig.setAuthType(jwtProvider.authType()).setCredentialsProvider(this.wrapWithTokenFederationIfEnabled(jwtProvider));
            } else {
                OAuthM2MServicePrincipalCredentialsProvider m2mProvider = new OAuthM2MServicePrincipalCredentialsProvider();
                this.databricksConfig.setAuthType("oauth-m2m").setCredentialsProvider(this.wrapWithTokenFederationIfEnabled(m2mProvider));
            }
        }
    }

    private void setupAzureMI() {
        this.databricksConfig.setHost(this.connectionContext.getHostForOAuth());
        this.databricksConfig.setAuthType("azure-msi");
        this.databricksConfig.setCredentialsProvider(this.wrapWithTokenFederationIfEnabled(new AzureMSICredentialProvider(this.connectionContext)));
    }

    public static String convertNonProxyHostConfigToBeSystemPropertyCompliant(String nonProxyHosts) {
        if (nonProxyHosts == null || nonProxyHosts.isEmpty()) {
            return "";
        }
        if (nonProxyHosts.contains("|")) {
            return nonProxyHosts;
        }
        return Arrays.stream(nonProxyHosts.split(",")).map(suffix -> {
            if (suffix.startsWith(".")) {
                return "*" + suffix;
            }
            return suffix;
        }).collect(Collectors.joining("|"));
    }

    public DatabricksConfig getDatabricksConfig() {
        return this.databricksConfig;
    }

    private void setupDiscoveryEndpoint() {
        if (this.connectionContext.isOAuthDiscoveryModeEnabled()) {
            this.databricksConfig.setDiscoveryUrl(this.connectionContext.getOAuthDiscoveryURL());
        }
    }

    private CredentialsProvider wrapWithTokenFederationIfEnabled(CredentialsProvider provider) {
        if (this.connectionContext.isTokenFederationEnabled()) {
            return new DatabricksTokenFederationProvider(this.connectionContext, provider);
        }
        return provider;
    }
}

