/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.jdbc.dbclient.impl.common;

import com.databricks.internal.apache.http.config.Registry;
import com.databricks.internal.apache.http.config.RegistryBuilder;
import com.databricks.internal.apache.http.conn.socket.ConnectionSocketFactory;
import com.databricks.internal.apache.http.conn.socket.PlainConnectionSocketFactory;
import com.databricks.internal.apache.http.conn.ssl.SSLConnectionSocketFactory;
import com.databricks.internal.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import com.databricks.jdbc.api.internal.IDatabricksConnectionContext;
import com.databricks.jdbc.common.util.SocketFactoryUtil;
import com.databricks.jdbc.exception.DatabricksSSLException;
import com.databricks.jdbc.log.JdbcLogger;
import com.databricks.jdbc.log.JdbcLoggerFactory;
import com.databricks.jdbc.model.telemetry.enums.DatabricksDriverErrorCode;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertPathValidator;
import java.security.cert.CertSelector;
import java.security.cert.CertificateException;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.PKIXRevocationChecker;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;
import javax.net.ssl.CertPathTrustManagerParameters;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

public class ConfiguratorUtils {
    private static final JdbcLogger LOGGER = JdbcLoggerFactory.getLogger(ConfiguratorUtils.class);
    private static final String JAVA_TRUST_STORE_PATH_PROPERTY = "javax.net.ssl.trustStore";
    private static final String JAVA_TRUST_STORE_PASSWORD_PROPERTY = "javax.net.ssl.trustStorePassword";
    private static final String JAVA_TRUST_STORE_TYPE_PROPERTY = "javax.net.ssl.trustStoreType";

    private static boolean isJDBCTestEnv() {
        return Boolean.parseBoolean(System.getenv("IS_JDBC_TEST_ENV"));
    }

    public static PoolingHttpClientConnectionManager getBaseConnectionManager(IDatabricksConnectionContext connectionContext) throws DatabricksSSLException {
        if (connectionContext.getSSLTrustStore() == null && connectionContext.checkCertificateRevocation() && !connectionContext.acceptUndeterminedCertificateRevocation() && !connectionContext.useSystemTrustStore() && !connectionContext.allowSelfSignedCerts()) {
            return new PoolingHttpClientConnectionManager();
        }
        if (ConfiguratorUtils.isJDBCTestEnv()) {
            LOGGER.info("Using trust-all socket factory for JDBC test environment");
            return new PoolingHttpClientConnectionManager(SocketFactoryUtil.getTrustAllSocketFactoryRegistry());
        }
        if (connectionContext.allowSelfSignedCerts()) {
            LOGGER.warn("Self-signed certificates are allowed. Please only use this parameter (AllowSelfSignedCerts) when you're sure of what you're doing. This is not recommended for production use.");
            return new PoolingHttpClientConnectionManager(SocketFactoryUtil.getTrustAllSocketFactoryRegistry());
        }
        Registry<ConnectionSocketFactory> socketFactoryRegistry = ConfiguratorUtils.createConnectionSocketFactoryRegistry(connectionContext);
        return new PoolingHttpClientConnectionManager(socketFactoryRegistry);
    }

    public static Registry<ConnectionSocketFactory> createConnectionSocketFactoryRegistry(IDatabricksConnectionContext connectionContext) throws DatabricksSSLException {
        if (connectionContext.getSSLTrustStore() != null) {
            return ConfiguratorUtils.createRegistryWithCustomTrustStore(connectionContext);
        }
        return ConfiguratorUtils.createRegistryWithSystemOrDefaultTrustStore(connectionContext);
    }

    private static Registry<ConnectionSocketFactory> createRegistryWithCustomTrustStore(IDatabricksConnectionContext connectionContext) throws DatabricksSSLException {
        try {
            Set<TrustAnchor> trustAnchors;
            KeyStore trustStore = ConfiguratorUtils.loadTruststoreOrNull(connectionContext);
            if (trustStore == null) {
                String errorMessage = "Specified trust store could not be loaded: " + connectionContext.getSSLTrustStore();
                LOGGER.debug("Trust store load failed: " + connectionContext.getSSLTrustStore());
                ConfiguratorUtils.handleError(errorMessage, new IOException(errorMessage));
            }
            if ((trustAnchors = ConfiguratorUtils.getTrustAnchorsFromTrustStore(trustStore)).isEmpty()) {
                String errorMessage = "Custom trust store contains no trust anchors. Certificate validation will fail.";
                ConfiguratorUtils.handleError(errorMessage, new CertificateException(errorMessage));
            }
            LOGGER.info("Using custom trust store: " + connectionContext.getSSLTrustStore());
            return ConfiguratorUtils.createRegistryFromTrustAnchors(trustAnchors, connectionContext, "custom trust store: " + connectionContext.getSSLTrustStore());
        }
        catch (Exception e) {
            ConfiguratorUtils.handleError("Error while setting up custom trust store: " + connectionContext.getSSLTrustStore(), e);
            return null;
        }
    }

    private static Registry<ConnectionSocketFactory> createRegistryWithSystemOrDefaultTrustStore(IDatabricksConnectionContext connectionContext) throws DatabricksSSLException {
        String sysTrustStore = null;
        if (connectionContext.useSystemTrustStore()) {
            sysTrustStore = System.getProperty(JAVA_TRUST_STORE_PATH_PROPERTY);
        }
        if (sysTrustStore != null && !sysTrustStore.isEmpty()) {
            return ConfiguratorUtils.createRegistryWithSystemPropertyTrustStore(connectionContext, sysTrustStore);
        }
        return ConfiguratorUtils.createRegistryWithJdkDefaultTrustStore(connectionContext);
    }

    private static Registry<ConnectionSocketFactory> createRegistryWithSystemPropertyTrustStore(IDatabricksConnectionContext connectionContext, String sysTrustStore) throws DatabricksSSLException {
        try {
            LOGGER.info("Using system property javax.net.ssl.trustStore: " + sysTrustStore + " (This overrides the JDK's default cacerts store)");
            File trustStoreFile = new File(sysTrustStore);
            if (!trustStoreFile.exists()) {
                String errorMessage = "System property trust store file does not exist: " + sysTrustStore;
                ConfiguratorUtils.handleError(errorMessage, new IOException(errorMessage));
            }
            KeyStore trustStore = KeyStore.getInstance(System.getProperty(JAVA_TRUST_STORE_TYPE_PROPERTY, "JKS"));
            char[] password = null;
            String passwordProp = System.getProperty(JAVA_TRUST_STORE_PASSWORD_PROPERTY);
            if (passwordProp != null) {
                password = passwordProp.toCharArray();
            }
            try (FileInputStream fis = new FileInputStream(sysTrustStore);){
                trustStore.load(fis, password);
            }
            Set<TrustAnchor> trustAnchors = ConfiguratorUtils.getTrustAnchorsFromTrustStore(trustStore);
            return ConfiguratorUtils.createRegistryFromTrustAnchors(trustAnchors, connectionContext, "system property trust store: " + sysTrustStore);
        }
        catch (DatabricksSSLException | IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            ConfiguratorUtils.handleError("Error while setting up system property trust store: " + sysTrustStore, e);
            return null;
        }
    }

    private static Registry<ConnectionSocketFactory> createRegistryWithJdkDefaultTrustStore(IDatabricksConnectionContext connectionContext) throws DatabricksSSLException {
        try {
            if (connectionContext.useSystemTrustStore()) {
                LOGGER.info("No system property trust store found, using JDK default trust store (cacerts)");
            } else {
                LOGGER.info("UseSystemTrustStore=false, using JDK default trust store (cacerts) and ignoring system properties");
            }
            Set<TrustAnchor> systemTrustAnchors = ConfiguratorUtils.getTrustAnchorsFromTrustStore(null);
            return ConfiguratorUtils.createRegistryFromTrustAnchors(systemTrustAnchors, connectionContext, "JDK default trust store (cacerts)");
        }
        catch (DatabricksSSLException e) {
            ConfiguratorUtils.handleError("Error while setting up JDK default trust store", e);
            return null;
        }
    }

    private static Registry<ConnectionSocketFactory> createRegistryFromTrustAnchors(Set<TrustAnchor> trustAnchors, IDatabricksConnectionContext connectionContext, String sourceDescription) throws DatabricksSSLException {
        if (trustAnchors == null || trustAnchors.isEmpty()) {
            throw new DatabricksSSLException(sourceDescription + " contains no trust anchors", DatabricksDriverErrorCode.SSL_HANDSHAKE_ERROR);
        }
        try {
            TrustManager[] trustManagers = ConfiguratorUtils.createTrustManagers(trustAnchors, connectionContext.checkCertificateRevocation(), connectionContext.acceptUndeterminedCertificateRevocation());
            KeyStore keyStore = ConfiguratorUtils.loadKeystoreOrNull(connectionContext);
            if (keyStore != null) {
                LOGGER.info("Client certificate authentication enabled");
                char[] keyStorePassword = null;
                if (connectionContext.getSSLKeyStorePassword() != null) {
                    keyStorePassword = connectionContext.getSSLKeyStorePassword().toCharArray();
                }
                KeyManager[] keyManagers = ConfiguratorUtils.createKeyManagers(keyStore, keyStorePassword);
                return ConfiguratorUtils.createSocketFactoryRegistry(trustManagers, keyManagers);
            }
            LOGGER.debug("No keystore path specified in connection url");
            return ConfiguratorUtils.createSocketFactoryRegistry(trustManagers);
        }
        catch (Exception e) {
            ConfiguratorUtils.handleError("Error setting up SSL socket factory for " + sourceDescription, e);
            return null;
        }
    }

    private static Registry<ConnectionSocketFactory> createSocketFactoryRegistry(TrustManager[] trustManagers) throws NoSuchAlgorithmException, KeyManagementException {
        return ConfiguratorUtils.createSocketFactoryRegistry(trustManagers, null);
    }

    private static KeyManager[] createKeyManagers(KeyStore keyStore, char[] keyStorePassword) throws NoSuchAlgorithmException, KeyStoreException, DatabricksSSLException {
        try {
            KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            kmf.init(keyStore, keyStorePassword);
            LOGGER.info("Successfully initialized key managers for client certificate authentication");
            return kmf.getKeyManagers();
        }
        catch (UnrecoverableKeyException e) {
            String errorMessage = "Failed to initialize key managers: " + e.getMessage();
            LOGGER.error(errorMessage);
            throw new DatabricksSSLException(errorMessage, (Throwable)e, DatabricksDriverErrorCode.SSL_HANDSHAKE_ERROR);
        }
    }

    private static Registry<ConnectionSocketFactory> createSocketFactoryRegistry(TrustManager[] trustManagers, KeyManager[] keyManagers) throws NoSuchAlgorithmException, KeyManagementException {
        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(keyManagers, trustManagers, new SecureRandom());
        SSLConnectionSocketFactory sslSocketFactory = new SSLConnectionSocketFactory(sslContext);
        return RegistryBuilder.create().register("https", sslSocketFactory).register("http", (SSLConnectionSocketFactory)((Object)new PlainConnectionSocketFactory())).build();
    }

    private static TrustManager[] createTrustManagers(Set<TrustAnchor> trustAnchors, boolean checkCertificateRevocation, boolean acceptUndeterminedCertificateRevocation) throws NoSuchAlgorithmException, InvalidAlgorithmParameterException, DatabricksSSLException {
        CertPathTrustManagerParameters trustManagerParams = ConfiguratorUtils.buildTrustManagerParameters(trustAnchors, checkCertificateRevocation, acceptUndeterminedCertificateRevocation);
        TrustManagerFactory customTmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        customTmf.init(trustManagerParams);
        LOGGER.info("Certificate revocation check: " + checkCertificateRevocation);
        return customTmf.getTrustManagers();
    }

    private static X509TrustManager findX509TrustManager(TrustManager[] trustManagers) {
        if (trustManagers == null) {
            return null;
        }
        for (TrustManager tm : trustManagers) {
            if (!(tm instanceof X509TrustManager)) continue;
            return (X509TrustManager)tm;
        }
        return null;
    }

    public static KeyStore loadTruststoreOrNull(IDatabricksConnectionContext connectionContext) throws DatabricksSSLException {
        KeyStore keyStore;
        String trustStorePath = connectionContext.getSSLTrustStore();
        if (trustStorePath == null) {
            LOGGER.debug("No truststore path specified in connection url");
            return null;
        }
        File trustStoreFile = new File(trustStorePath);
        if (!trustStoreFile.exists()) {
            String errorMessage = "Specified trust store file does not exist: " + trustStorePath;
            LOGGER.error(errorMessage);
            throw new DatabricksSSLException(errorMessage, DatabricksDriverErrorCode.SSL_HANDSHAKE_ERROR);
        }
        char[] password = null;
        if (connectionContext.getSSLTrustStorePassword() != null) {
            password = connectionContext.getSSLTrustStorePassword().toCharArray();
        }
        String trustStoreType = connectionContext.getSSLTrustStoreType();
        String trustStoreProvider = connectionContext.getSSLTrustStoreProvider();
        FileInputStream trustStoreStream = new FileInputStream(trustStorePath);
        try {
            KeyStore trustStore;
            LOGGER.info("Loading trust store as type: " + trustStoreType);
            if (trustStoreProvider != null && !trustStoreProvider.isEmpty()) {
                LOGGER.info("Using trust store provider: " + trustStoreProvider);
                trustStore = KeyStore.getInstance(trustStoreType, trustStoreProvider);
            } else {
                trustStore = KeyStore.getInstance(trustStoreType);
            }
            trustStore.load(trustStoreStream, password);
            LOGGER.info("Successfully loaded trust store: " + trustStorePath);
            keyStore = trustStore;
        }
        catch (Throwable trustStore) {
            try {
                try {
                    trustStoreStream.close();
                }
                catch (Throwable throwable) {
                    trustStore.addSuppressed(throwable);
                }
                throw trustStore;
            }
            catch (Exception e) {
                String errorMessage = trustStoreProvider != null && !trustStoreProvider.isEmpty() ? String.format("Failed to load trust store: %s with type %s and provider %s: %s", trustStorePath, trustStoreType, trustStoreProvider, e.getMessage()) : String.format("Failed to load trust store: %s with type %s: %s", trustStorePath, trustStoreType, e.getMessage());
                LOGGER.error(errorMessage.toString());
                throw new DatabricksSSLException(errorMessage.toString(), (Throwable)e, DatabricksDriverErrorCode.SSL_HANDSHAKE_ERROR);
            }
        }
        trustStoreStream.close();
        return keyStore;
    }

    public static KeyStore loadKeystoreOrNull(IDatabricksConnectionContext connectionContext) throws DatabricksSSLException {
        String keyStorePath = connectionContext.getSSLKeyStore();
        if (keyStorePath == null) {
            LOGGER.debug("No keystore path specified in connection context");
            return null;
        }
        File keyStoreFile = new File(keyStorePath);
        if (!keyStoreFile.exists()) {
            String errorMessage = "Specified key store file does not exist: " + keyStorePath;
            LOGGER.error(errorMessage);
            throw new DatabricksSSLException(errorMessage, DatabricksDriverErrorCode.SSL_HANDSHAKE_ERROR);
        }
        String keyStorePassword = connectionContext.getSSLKeyStorePassword();
        if (keyStorePassword == null) {
            String errorMessage = "Key store password is required when a key store is specified: " + keyStorePath;
            LOGGER.error(errorMessage);
            throw new DatabricksSSLException(errorMessage, DatabricksDriverErrorCode.SSL_HANDSHAKE_ERROR);
        }
        char[] password = keyStorePassword.toCharArray();
        String keyStoreType = connectionContext.getSSLKeyStoreType();
        String keyStoreProvider = connectionContext.getSSLKeyStoreProvider();
        try {
            KeyStore keyStore;
            LOGGER.info("Loading key store as type: " + keyStoreType);
            if (keyStoreProvider != null && !keyStoreProvider.isEmpty()) {
                LOGGER.info("Using key store provider: " + keyStoreProvider);
                keyStore = KeyStore.getInstance(keyStoreType, keyStoreProvider);
            } else {
                keyStore = KeyStore.getInstance(keyStoreType);
            }
            try (FileInputStream keyStoreStream = new FileInputStream(keyStorePath);){
                keyStore.load(keyStoreStream, password);
            }
            LOGGER.debug("Successfully loaded key store: " + keyStorePath);
            return keyStore;
        }
        catch (Exception e) {
            StringBuilder errorMessage = new StringBuilder().append("Failed to load key store: ").append(keyStorePath).append(" with type ").append(keyStoreType).append(": ").append(e.getMessage());
            LOGGER.error(errorMessage.toString(), e);
            throw new DatabricksSSLException(errorMessage.toString(), (Throwable)e, DatabricksDriverErrorCode.SSL_HANDSHAKE_ERROR);
        }
    }

    public static Set<TrustAnchor> getTrustAnchorsFromTrustStore(KeyStore trustStore) throws DatabricksSSLException {
        try {
            TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(trustStore);
            TrustManager[] trustManagers = trustManagerFactory.getTrustManagers();
            X509TrustManager x509TrustManager = ConfiguratorUtils.findX509TrustManager(trustManagers);
            if (x509TrustManager == null || x509TrustManager.getAcceptedIssuers().length == 0) {
                return Collections.emptySet();
            }
            return Arrays.stream(x509TrustManager.getAcceptedIssuers()).map(cert -> new TrustAnchor((X509Certificate)cert, null)).collect(Collectors.toSet());
        }
        catch (KeyStoreException | NoSuchAlgorithmException e) {
            ConfiguratorUtils.handleError("Error while getting trust anchors from trust store: " + e.getMessage(), e);
            return Collections.emptySet();
        }
    }

    public static CertPathTrustManagerParameters buildTrustManagerParameters(Set<TrustAnchor> trustAnchors, boolean checkCertificateRevocation, boolean acceptUndeterminedCertificateRevocation) throws DatabricksSSLException {
        try {
            PKIXBuilderParameters pkixBuilderParameters = new PKIXBuilderParameters(trustAnchors, (CertSelector)new X509CertSelector());
            pkixBuilderParameters.setRevocationEnabled(checkCertificateRevocation);
            if (checkCertificateRevocation) {
                CertPathValidator certPathValidator = CertPathValidator.getInstance("PKIX");
                PKIXRevocationChecker revocationChecker = (PKIXRevocationChecker)certPathValidator.getRevocationChecker();
                if (acceptUndeterminedCertificateRevocation) {
                    revocationChecker.setOptions(Set.of(PKIXRevocationChecker.Option.SOFT_FAIL, PKIXRevocationChecker.Option.NO_FALLBACK, PKIXRevocationChecker.Option.PREFER_CRLS));
                }
                LOGGER.info("Certificate revocation enabled. Undetermined revocation accepted: " + acceptUndeterminedCertificateRevocation);
                pkixBuilderParameters.addCertPathChecker(revocationChecker);
            }
            return new CertPathTrustManagerParameters(pkixBuilderParameters);
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException e) {
            ConfiguratorUtils.handleError("Error while building trust manager parameters: " + e.getMessage(), e);
            return null;
        }
    }

    private static void handleError(String errorMessage, Exception e) throws DatabricksSSLException {
        LOGGER.error(errorMessage, e);
        throw new DatabricksSSLException(errorMessage, (Throwable)e, DatabricksDriverErrorCode.SSL_HANDSHAKE_ERROR);
    }
}

